package img.io;

import java.awt.Color;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOError;
import java.io.IOException;

/**
 * Classe utilitaire pour la lecture et l'écriture de fichiers images.
 */
public final class ImageIO {

	/**
	 * Unique constructeur privé destiné à interdire l'instanciation de la classe.
	 */
	private ImageIO() {
	}

	/**
	 * Paramètre des notifications sur la console.
	 */
	private static int verbose = 0;

	/**
	 * Indique si les actions doivent êtres notifiées sur la console par une valeur
	 * entière : 0 = aucune ; 1 = chargement ; 2 = sauvegarde ; 3 = les deux
	 * 
	 * @param n paramètre des notifications
	 */
	public static void verbose(int n) {
		verbose = (n < 0 ? 0 : (n > 3 ? 3 : n));
	}

	/**
	 * Chargement d'un fichier image dans une matrice de pixels.
	 * 
	 * @param fichier nom du fichier à charger
	 * @return matrice de pixels
	 */
	public static Color[][] load(String fichier) {
		if (verbose % 2 == 1)
			System.out.print("Chargement de " + fichier + "... ");
		BufferedImage image;
		try {
			image = javax.imageio.ImageIO.read(new File(fichier));
		} catch (IOException e) {
			throw new IOError(e);
		}
		Color[][] pixels;
		pixels = new Color[image.getHeight()][image.getWidth()];
		for (int y = 0; y < image.getHeight(); y++)
			for (int x = 0; x < image.getWidth(); x++)
				pixels[y][x] = new Color(image.getRGB(x, y));
		if (verbose % 2 == 1)
			System.out.println("ok.");
		return pixels;
	}

	/**
	 * Sauvegarde d'une matrice de pixels dans un fichier image.
	 * 
	 * @param pixels  matrice de pixels à sauvegarder
	 * @param fichier nom du fichier à créer ou à mettre à jour
	 */
	public static void save(Color[][] pixels, String fichier) {
		if (verbose > 1)
			System.out.print("Sauvegarde dans " + fichier + "... ");
		int i = fichier.lastIndexOf('.');
		String type = (i > 0 ? fichier.substring(i + 1) : "png");
		BufferedImage image = new BufferedImage(pixels[0].length, pixels.length, BufferedImage.TYPE_INT_RGB);
		for (int y = 0; y < pixels.length; y++)
			for (int x = 0; x < pixels[y].length; x++)
				image.setRGB(x, y, pixels[y][x].getRGB());
		try {
			javax.imageio.ImageIO.write(image, type, new File(fichier));
		} catch (IOException e) {
			throw new IOError(e);
		}
		if (verbose > 1)
			System.out.println("ok.");
	}

}
